/**
 * @brief Module System - Generic system support functions
 *
 * This module contains generic functions to support the system and
 * don't belong to any other module.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/module_system.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "module_system.h"
#include "class_config.h"
#include "support.h"

#define SECTION "SYSTEM"

struct moddata_system {
	char *userallowed;  /* remember system tag for config file saving */
	int  autorescan;
	int  rescantimer;
} modbase_system;

int
system_init ()
{
	struct moddata_system *base = &modbase_system;
	int val;

	base->rescantimer = 0;
	base->userallowed = config_get_string  (SECTION, "userallowed", "");
	base->autorescan  = config_get_boolean (SECTION, "autorescan", TRUE);
	val               = config_get_int     (SECTION, "CmdTimeout", 8);
	set_timeforcmd(val);

	ipc_protect_tag (TAG_SAVECONFIG);  /* tags only for privileged use */

	register_function (T1000QUEUE, system_timer1000);
	register_function (QUERYQUEUE, system_query);
	register_function (CONFIGQUEUE, system_configure);
	return 0;
}

void
system_exit ()
{
	struct moddata_system *base = &modbase_system;
	
	g_free (base->userallowed);
}

void
system_query (struct tagitem *taglist)
{
	system_handle_tags (MODE_QUERY, taglist);
}

void
system_configure (struct tagitem *taglist)
{
	system_handle_tags (MODE_CONFIG, taglist);
}

void
system_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_system *base = &modbase_system;
	struct tagitem args[] = {{ TAG_AMBIENTLIGHT, 0 },
	                         { TAG_SLEEPSUPPORTED, 0 },
	                         { TAG_SYSINFOSLAVE, 0 },
	                         { TAG_END, 0 }};
	static char *version = VERSION;
	int val;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_SAVECONFIG:
			if (cfgure) {
				if (!config_save ())
					tagerror (taglist, E_NOSUPPORT);
			} else
				tagerror (taglist, E_NOREAD);
			break;
		case TAG_USERALLOWED:  /* private tag */
			if (cfgure)	{
				replace_string (&base->userallowed, (char *) taglist->data);
				config_set_string (SECTION, "userallowed", base->userallowed);
			} else if (strlen (base->userallowed) > 0)
				taglist->data = (long) base->userallowed;
			break;
		case TAG_VERSION:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = (long) version;
			break;
		case TAG_AUTORESCAN:
			if (cfgure)	{
				base->autorescan = taglist->data;
				config_set_boolean (SECTION, "autorescan", taglist->data);
			} else
				taglist->data = (long) base->autorescan;
			break;
		case TAG_TIMEFORCMD:
			if (cfgure)	{
				set_timeforcmd(taglist->data);
				config_set_int (SECTION, "CmdTimeout", taglist->data);
			} else
				taglist->data = (long) get_timeforcmd();
			break;
		case TAG_SYSINFO:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else {
				val = 0;
				if (config_iswritable())
					val |= SYSINFO_CONFIGWRITABLE;

				process_queue (QUERYQUEUE, args);
				if (tagfind (args, TAG_AMBIENTLIGHT, -1) != (tag_t) -1)
					val |= SYSINFO_HAVELMU;
				if (tagfind (args, TAG_SLEEPSUPPORTED, -1) != (tag_t) -1)
					val |= SYSINFO_SLEEPSUPPORTED;

				/* TAG_SYSINFOSLAVE asks other modules if there are additiononal
				 * machine dependent information that should be added. Currently
				 * only module_pmac react to that.
				 * */
				val |= tagfind (args, TAG_SYSINFOSLAVE, 0);
#ifdef WITH_PMUD
				val |= SYSINFO_PMUDSUPPORT;
#endif
#ifdef WITH_OSS
				val |= SYSINFO_HAVEOSS;
#endif
#ifdef WITH_ALSA
				val |= SYSINFO_HAVEALSA;
#endif
#ifdef WITH_IBAM
				val |= SYSINFO_HAVEIBAM;
#endif
				taglist->data = val;
			}
			break;
		}
		taglist++;
	}
}

void
system_timer1000 (struct tagitem *taglist)
{
	struct moddata_system *base = &modbase_system;

	if (base->autorescan) {
		base->rescantimer++;
		if (base->rescantimer > RESCANTIME) {
			base->rescantimer = 0;
			scanEventDevices ();
		}
	}
}

