/*
 *  Jajuk
 *  Copyright (C) 2003-2011 The Jajuk Team
 *  http://jajuk.info
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *  $Revision$
 */
package ext.services.xml;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.jajuk.JUnitHelpers;
import org.jajuk.JajukTestCase;
import org.jajuk.util.log.Log;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * DOCUMENT_ME.
 */
public class TestXMLUtils extends JajukTestCase {

  // helper method to emma-coverage of the unused constructor
  /**
   * Test serializable utilities private constructor.
   * DOCUMENT_ME
   *
   * @throws Exception the exception
   */
  public void testSerializableUtilitiesPrivateConstructor() throws Exception {
    // For EMMA code-coverage tests
    JUnitHelpers.executePrivateConstructor(XMLUtils.class);
  }

  /**
   * Test method for.
   *
   * {@link ext.XMLUtils#getChildElement(org.w3c.dom.Element, java.lang.String)}
   * .
   */
  public void testGetChildElement() {
    Document doc = XMLUtils.getDocument("<xml><test/></xml>");
    assertNotNull(doc);

    assertNotNull(XMLUtils.getChildElement(doc.getDocumentElement(), "test"));
  }

  /**
   * Test get child element null.
   * DOCUMENT_ME
   */
  public void testGetChildElementNull() {
    assertNull(XMLUtils.getChildElement(null, "test"));
  }

  /**
   * Test get child element not existing.
   * DOCUMENT_ME
   */
  public void testGetChildElementNotExisting() {
    Document doc = XMLUtils.getDocument("<xml><test/></xml>");
    assertNotNull(doc);

    assertNull(XMLUtils.getChildElement(doc.getDocumentElement(), "notexist"));
  }

  /**
   * Test method for.
   *
   * {@link ext.XMLUtils#getAttributeValue(org.w3c.dom.Element, java.lang.String)}
   * .
   */
  public void testGetAttributeValue() {
    Document doc = XMLUtils.getDocument("<xml><test value=\"1\"/></xml>");
    assertNotNull(doc);

    Element test = XMLUtils.getChildElement(doc.getDocumentElement(), "test");
    assertNotNull(test);

    assertNotNull(XMLUtils.getAttributeValue(test, "value"));
    assertEquals("", XMLUtils.getAttributeValue(test, "notexist"));
    assertEquals("", XMLUtils.getAttributeValue(doc.getDocumentElement(), "value"));
  }

  /**
   * Test get attribute value null.
   * DOCUMENT_ME
   */
  public void testGetAttributeValueNull() {
    assertNull(XMLUtils.getAttributeValue(null, "value"));
  }

  /**
   * Test method for.
   *
   * {@link ext.XMLUtils#getChildElementContent(org.w3c.dom.Element, java.lang.String)}
   * .
   */
  public void testGetChildElementContent() {
    Document doc = XMLUtils.getDocument("<xml><test value=\"1\">testcontent</test></xml>");
    assertNotNull(doc);

    assertEquals("testcontent", XMLUtils.getChildElementContent(doc.getDocumentElement(), "test"));
  }

  /**
   * Test get child element content null.
   * DOCUMENT_ME
   */
  public void testGetChildElementContentNull() {
    assertEquals("", XMLUtils.getChildElementContent(null, "test"));
  }

  /**
   * Test method for {@link ext.XMLUtils#getDocument(java.lang.String)}.
   */

  public void testGetDocument() {
    Document doc = XMLUtils.getDocument("<xml><test value=\"1\"/></xml>");
    assertNotNull(doc);

    assertEquals("xml", doc.getDocumentElement().getTagName());
  }

  /**
   * Test get document parse error.
   * DOCUMENT_ME
   */
  public void testGetDocumentParseError() {
    Document doc = XMLUtils.getDocument("<xmlinvalid>adsasd<asdksdtest value=\"1\"/></xml>");
    assertNull(doc);
    doc = XMLUtils.getDocument(StringUtils.repeat("1", 1000)); // more than 500
    // characters for
    // log.debug
    assertNull(doc);
  }

  /**
   * Test method for.
   *
   * @throws Exception the exception
   * {@link ext.services.xml.XMLUtils#writeBeanToFile(java.lang.Object, java.lang.String)}
   * .
   */
  public final void testWriteBeanToFile() throws Exception {

    PersonBean bean = new PersonBean();
    bean.setName("testvalue");

    File file = File.createTempFile("test", ".bean");
    assertTrue(file.delete()); // delte file to create it from scratch
    assertFalse(file.exists());
    XMLUtils.writeBeanToFile(bean, file.getAbsolutePath());

    { // check file
      assertTrue(file.exists());
      String contents = FileUtils.readFileToString(file);
      Log.debug("Contents: " + contents);
      assertTrue(contents, file.length() > 0);
    }

    /*
     * TODO: somehow this test does not work, we should find out and fix this
     * test...
     * 
     * Object obj = XMLUtils.readBeanFromFile(file.getAbsolutePath());
     * assertTrue(obj instanceof PersonBean);
     * 
     * assertEquals("testvalue", ((PersonBean)obj).getName());
     */
  }

  /**
   * Test write bean to file invalid file.
   * DOCUMENT_ME
   */
  public final void testWriteBeanToFileInvalidFile() {

    PersonBean bean = new PersonBean();
    bean.setName("testvalue");

    try {
      XMLUtils.writeBeanToFile(bean,
          "&@#$@(*^)!#!@#@#)}{?M<>?<?,/.,/.,\"'\\][|}{{:2zqwq;sInvalidFileName/\\/");
      fail("Should throw Exception here...");
    } catch (IOException e) {
      // expected...
    }
  }

  /**
   * Class <code>PersonBean</code>.
   */
  public class PersonBean implements java.io.Serializable {
    
    /** Generated serialVersionUID. */
    private static final long serialVersionUID = 1124123276327532379L;

    /** DOCUMENT_ME. */
    private String name;

    /** DOCUMENT_ME. */
    private boolean deceased;

    /** No-arg constructor (takes no arguments). */
    public PersonBean() {
    }

    /**
     * Property <code>name</code> (note capitalization) readable/writable.
     *
     * @return the name
     */
    public String getName() {
      return this.name;
    }

    /**
     * Setter for property <code>name</code>.
     *
     * @param name the new name
     */
    public void setName(final String name) {
      this.name = name;
    }

    /**
     * Getter for property "deceased" Different syntax for a boolean field (is
     * vs. get)
     *
     * @return true, if is deceased
     */
    public boolean isDeceased() {
      return this.deceased;
    }

    /**
     * Setter for property <code>deceased</code>.
     *
     * @param deceased the new deceased
     */
    public void setDeceased(final boolean deceased) {
      this.deceased = deceased;
    }
  }

  /**
   * Test method for.
   *
   * {@link ext.services.xml.XMLUtils#readBeanFromFile(java.lang.String)}.
   */
  public final void testReadBeanFromFile() {
    // tested above
  }

  /**
   * Test method for.
   *
   * {@link ext.services.xml.XMLUtils#readObjectFromFile(java.lang.String)}.
   */
  public final void testReadObjectFromFile() {
    // tested above
  }

  /**
   * Test method for.
   *
   * @throws Exception the exception
   * {@link ext.services.xml.XMLUtils#readObjectFromString(java.lang.String)}.
   */
  public final void testReadObjectFromString() throws Exception {
    String str = new String("teststring");

    File file = File.createTempFile("test", ".bean");
    assertTrue(file.delete()); // delte file to create it from scratch
    assertFalse(file.exists());
    XMLUtils.writeObjectToFile(str, file.getAbsolutePath());

    assertTrue(file.exists());

    String xml = FileUtils.readFileToString(file);

    assertTrue(StringUtils.isNotBlank(xml));
    assertNotNull(XMLUtils.getDocument(xml));

    /*
     * TODO: currently this reports an error about xpp3 pull parser missing, not
     * sure how this works inside Jajuk...
     * 
     * Object obj = XMLUtils.readObjectFromString(xml); assertTrue(obj
     * instanceof String);
     * 
     * assertEquals("teststring", obj);
     */
  }

  /**
   * Test method for.
   *
   * @throws Exception the exception
   * {@link ext.services.xml.XMLUtils#writeObjectToFile(java.lang.Object, java.lang.String)}
   * .
   */
  public final void testWriteObjectToFile() throws Exception {
    String str = new String("teststring");

    File file = File.createTempFile("test", ".bean");
    assertTrue(file.delete()); // delte file to create it from scratch
    assertFalse(file.exists());
    XMLUtils.writeObjectToFile(str, file.getAbsolutePath());

    assertTrue(file.exists());

    /*
     * TODO: currently this reports an error about xpp3 pull parser missing, not
     * sure how this works inside Jajuk...
     * 
     * Object obj = XMLUtils.readObjectFromFile(file.getAbsolutePath());
     * assertTrue(obj instanceof String);
     * 
     * assertEquals("teststring", obj);
     */
  }

  /**
   * Test write object to file invalid file.
   * DOCUMENT_ME
   */
  public final void testWriteObjectToFileInvalidFile() {
    String str = new String("teststring");

    try {
      XMLUtils.writeObjectToFile(str,
          "&@#$@(*^)!#!@#@#)}{?M<>?<?,/.,/.,\"'\\][|}{{:2zqwq;sInvalidFileName/\\/");
      fail("Should throw Exception here...");
    } catch (IOException e) {
      // expected...
    }
  }
}
