// ---------------------------------------------------------------------------
// - TcpServer.cpp                                                           -
// - afnix:net module - tcp server socket implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Integer.hpp"
#include "TcpServer.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // default tcp backlog
  const long AFNIX_TCP_BACKLOG = 5;

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a tcp server on a ephemeral port
  
  TcpServer::TcpServer (void) {
    // set default backlog
    d_blog = AFNIX_TCP_BACKLOG;
    // bind and listen this socket
    if (bind (0) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (d_blog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with a port

  TcpServer::TcpServer (t_word port) {
    // set default backlog
    d_blog = AFNIX_TCP_BACKLOG;
    // bind and listen this socket
    if (bind (port) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (d_blog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with a port and backlog

  TcpServer::TcpServer (t_word port, const long backlog) {
    // set backlog
    d_blog = backlog > 0 ? backlog : AFNIX_TCP_BACKLOG;
    // bind and listen this socket
    if (bind (port) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (d_blog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with a host and port

  TcpServer::TcpServer (const String& host, t_word port) : TcpSocket (false) {
    // set default backlog
    d_blog = AFNIX_TCP_BACKLOG;
    // get the host address
    Address addr (host);
    // create athe socket
    create (addr);
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (d_blog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with an address and port

  TcpServer::TcpServer (const Address& addr, t_word port) : TcpSocket (false) {
    // set default backlog
    d_blog = AFNIX_TCP_BACKLOG;
    // create the socket
    create (addr);
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (d_blog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with a host and port and backlog

  TcpServer::TcpServer (const String& host, t_word port, 
			const long backlog) : TcpSocket (false) {
    // set backlog
    d_blog = backlog > 0 ? backlog : AFNIX_TCP_BACKLOG;
    // get the host address
    Address addr (host);
    // create the socket
    create (addr);
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (d_blog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with an address, a port and backlog

  TcpServer::TcpServer (const Address& addr, t_word port, 
			const long backlog) : TcpSocket (false) {
    // set backlog
    d_blog = backlog > 0 ? backlog : AFNIX_TCP_BACKLOG;
    // create the socket
    create (addr);
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (d_blog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // return the class name

  String TcpServer::repr (void) const {
    return "TcpServer";
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // create a new object in a generic way

  Object* TcpServer::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 arguments
    if (argc == 0) return new TcpServer;
    // check for 1 argument
    if (argc == 1) {
      t_word port = argv->getlong (0);
      return new TcpServer (port);
    }
    // check for 2 arguments
    if (argc == 2) {
      Object* obj = argv->get (0);
      // check for a port
      Integer* iobj = dynamic_cast <Integer*> (obj);
      if (iobj != nilp) {
	t_word port    = iobj->tolong ();
	long   backlog = argv->getlong (1);
	return new TcpServer (port, backlog);
      }
      // check for a host
      String* sobj = dynamic_cast <String*> (obj);
      if (sobj != nilp) {
	t_word port    = argv->getlong (1);
	return new TcpServer (*sobj, port);
      }
      // check for an address
      Address* aobj = dynamic_cast <Address*> (obj);
      if (aobj != nilp) {
	t_word port = argv->getlong (1);
	return new TcpServer (*aobj, port);
      }
    }
    // check for 3 arguments
    if (argc == 3) {
      Object* obj = argv->get (0);
      // check for a host
      String* sobj = dynamic_cast <String*> (obj);
      if (sobj != nilp) {
	t_word port    = argv->getlong (1);
	long   backlog = argv->getlong (2);
	return new TcpServer (*sobj, port, backlog);
      }
      // check for an address
      Address* aobj = dynamic_cast <Address*> (obj);
      if (aobj != nilp) {
	t_word port    = argv->getlong (1);
	long   backlog = argv->getlong (2);
	return new TcpServer (*aobj, port, backlog);
      }
    }
    throw Exception ("argument-error", "invalid arguments with tcp server");
  }
}
