/* $Id: collision.c,v 1.67 2003/06/17 14:45:56 bsenders Exp $ */

#ifdef HAVE_GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "collision.h"
#include "list.h"
#include "model.h"
#include "object.h"
#include "player.h"
#include "food.h"

#define MDEBUG(...) DEBUG(DMODEL, "Model", __VA_ARGS__)

#define RE_NOCHECK  (0)
#define RE_UPLAYERS (1<<0)
#define RE_FOOD     (1<<1)
#define RE_BOXES    (1<<2)
#define RE_PLAYERS  (1<<3)

/*****************************************************************************
 * Rectangle ADT                                                             *
 *****************************************************************************/

typedef struct {
  int32_t x1, y1, x2, y2;
} Rect;

/* Create a new rectangle from given coordinates and size. */
static inline void
new_rect_from_xywh(Rect *r, int32_t x, int32_t y, int32_t w, int32_t h) {
  r->x1 = x;
  r->y1 = y;
  r->x2 = x + w - 1;
  r->y2 = y + h - 1;
}

/* Create a new rectangle from the current position of an object, and
 * return it. */
static inline void
new_rect_from_object_cur(Rect *r, Model_object *o) {
  new_rect_from_xywh(r, o->pos->x, o->pos->y, o->size->x, o->size->y);
}

/* Create a new rectangle from the desired position of an object, and
 * return it. */
static inline void
new_rect_from_object_des(Rect *r, Model_object *o) {
  new_rect_from_xywh(r, o->des_pos->x, o->des_pos->y, o->size->x, o->size->y);
}

/* Return TRUE if given rectangles overlap horizontally; FALSE otherwise. */
static inline int
rects_overlap_h(Rect *a, Rect *b) {
  return (b->y1 <= a->y1 && a->y1 <= b->y2) ||
         (a->y1 <= b->y1 && b->y1 <= a->y2);
}

/* Return TRUE if given rectangles overlap vertically; FALSE otherwise. */
static inline int
rects_overlap_v(Rect *a, Rect *b) {
  return (b->x1 <= a->x1 && a->x1 <= b->x2) ||
         (a->x1 <= b->x1 && b->x1 <= a->x2);
}

/* Return TRUE if given rectangles collide; FALSE otherwise. */
static inline int
rects_collide(Rect *a, Rect *b) {
  return rects_overlap_h(a, b) && rects_overlap_v(a, b);
}

/* Print characteristics of intersection of two given rectangles. */
static inline void
rect_intersect_debuginfo(Rect *a, Rect *b) {
  int32_t x1 = MAX(a->x1, b->x1);
  int32_t y1 = MAX(a->y1, b->y1);
  int32_t x2 = MIN(a->x2, b->x2);
  int32_t y2 = MIN(a->y2, b->y2);
  MDEBUG("assert: Intersection: <rectangle (%3d, %3d)-(%3d, %3d) (%3dx%3d)>",
         x1, y1, x2, y2, x2 - x1 + 1, y2 - y1 + 1);
}

/*****************************************************************************
 * Implementation of collides_with_objects                                   *
 *****************************************************************************/

/* Function tries to detect if rectangle around object will collide with its
 * surroundings.  Returns TRUE or FALSE whether this is not the case or is the
 * case respectively. */
static int
object_no_collides(void *other, void *newrect) {
  Model_object *o = (Model_object *)other;
  Rect otherrect;

  if (IS_DELETED(o))
    return TRUE;

  new_rect_from_object_cur(&otherrect, o);
  return !rects_collide(&otherrect, (Rect *)newrect);
}

/* Return TRUE if an object with specified position and size would collide with
 * any object in the specified list; FALSE otherwise. */
int
collides_with_objects(List *l, Vector *pos, Vector *size) {
  Rect newrect;

  new_rect_from_xywh(&newrect, pos->x, pos->y, size->x, size->y);
  return !list_foreach(l, object_no_collides, &newrect);
}

/*****************************************************************************
 * Implementation of has_no_floor                                            *
 *****************************************************************************/

static int
objobjnocol(void *obj1, void *obj2) {
  Model_object *o1 = (Model_object *)obj1;
  Model_object *o2 = (Model_object *)obj2;
  Rect a, b;
  
  if (IS_DELETED(o1) || IS_DELETED(o2) || o1 == o2)
    return TRUE;

  new_rect_from_object_cur(&a, o1);
  new_rect_from_object_cur(&b, o2);
  return !rects_collide(&a, &b);
}

static int
list_col_with_obj(List *l, Model_object *o) {
  return !list_foreach(l, objobjnocol, o);
}

/* Return TRUE if specified object has no floor (or other standable object)
 * underneath, FALSE otherwise. */
int
has_no_floor(Model *m, Model_object *o) {
  int res;
  assert(vectors_equal(o->pos, o->des_pos));
  o->pos->y += SGN(o->gravity->y);
  res = (!list_col_with_obj(m->boxes,        o) &&
         !list_col_with_obj(m->pas_food,     o) &&
         !list_col_with_obj(m->act_food,     o) &&
         !list_col_with_obj(m->act_uplayers, o) &&
         !list_col_with_obj(m->pas_uplayers, o));
  o->pos->y = o->des_pos->y;
  return res;
}

/*****************************************************************************
 * Collision resolvance functions                                            *
 *****************************************************************************/

/* Function to be called on collision from o1 with o2. */
typedef void (*Collide_func) (Model *m, Model_object *o1, Model_object *o2);

/* Move left and right away from each other horizontally, where distance is the
 * amount they should be moved away from each other. */
static void
move_away_h(Model_object *left, Model_object *right, int32_t distance) {
  int32_t total_speed = abs(left->speed->x) + abs(right->speed->x);
  int32_t left_distance, right_distance;
  MDEBUG("Horizontal collision %s<->%s", object_type_string[left->type],
                                         object_type_string[right->type]);
  assert(distance > 0);
  if (total_speed == 0) {
    left->des_pos->x  = left->pos->x;
    right->des_pos->x = right->pos->x;
  } else {
    left_distance  = (distance * abs(left->speed->x))  / total_speed;
    right_distance = (distance * abs(right->speed->x)) / total_speed;
    if (left_distance + right_distance != distance) {
      if (left->speed->x >= right->speed->x)
        left_distance++;
      else
        right_distance++;
    }

    left->des_pos->x  -= left_distance;
    right->des_pos->x += right_distance;

    left->speed->x  *= -(left->bounciness->x  / (float)MODEL_X_SCALE);
    right->speed->x *= -(right->bounciness->x / (float)MODEL_X_SCALE);
    left->accel->x  = 0;
    right->accel->x = 0;
  }

  left->accel->y  = left->gravity->y;
  right->accel->y = right->gravity->y;
}

/* Move top and bottom away from each other vertically, where distance is the
 * amount they should be moved away from each other. */
static void
move_away_v(Model_object *top, Model_object *bottom, int32_t distance) {
  int32_t total_speed = abs(top->speed->y) + abs(bottom->speed->y);
  int32_t top_distance, bottom_distance;
  MDEBUG("Vertical collision %s<->%s", object_type_string[top->type],
                                       object_type_string[bottom->type]);
  assert(distance > 0);
  if (total_speed == 0) {
    top->des_pos->y    = top->pos->y;
    bottom->des_pos->y = bottom->pos->y;
  } else {
    top_distance    = (distance * abs(top->speed->y))    / total_speed;
    bottom_distance = (distance * abs(bottom->speed->y)) / total_speed;
    if (top_distance + bottom_distance != distance) {
      if (top->speed->y >= bottom->speed->y)
        top_distance++;
      else
        bottom_distance++;
    }

    top->des_pos->y    -= top_distance;
    bottom->des_pos->y += bottom_distance;

    top->speed->y    *= -(top->bounciness->y    / (float)MODEL_Y_SCALE);
    bottom->speed->y *= -(bottom->bounciness->y / (float)MODEL_Y_SCALE);

    /* Pull speed of top object to zero if it has become very low, so not
     * constantly bouncing while walking */
    if (abs(top->speed->y) <= OBJ_Y_BOUNCE_CUTOFF) top->speed->y = 0;
  }
  /* Make sure one of the objects knows it stands on the other.
   * FIXME: Maybe is_standing should be replaced by a pointer to the object it
   * is standing on, so that for example different friction can be applied to
   * moving objects depending on the type of object they are standing on (so
   * one could have ice boxes and tar boxes). */
  top->is_standing = TRUE;
}

/* Resolve collision by moving objects away from each other. */
static void
colr_move_away(Model *m, Model_object *o1, Model_object *o2) {
  Rect cur_r1, cur_r2;
  Rect des_r1, des_r2;
  Rect newdes_r1, newdes_r2;
  int32_t h_overlap, v_overlap;
  int32_t h_distance, v_distance;
  Model_object *left, *right, *top, *bottom;

  new_rect_from_object_cur(&cur_r1, o1);
  new_rect_from_object_cur(&cur_r2, o2);
  new_rect_from_object_des(&des_r1, o1);
  new_rect_from_object_des(&des_r2, o2);

  h_overlap = rects_overlap_h(&cur_r1, &cur_r2);
  v_overlap = rects_overlap_v(&cur_r1, &cur_r2);

  if(h_overlap && v_overlap) {
    MDEBUG("assert: there is a current collision!");
    object_dump(o1, "assert: ");
    object_dump(o2, "assert: ");
    rect_intersect_debuginfo(&cur_r1, &cur_r2);
    assert(!h_overlap || !v_overlap);
  }

  if (cur_r1.x1 < cur_r2.x1) {
    h_distance = des_r1.x2 - des_r2.x1 + 1;
    left = o1, right = o2;
  } else {
    h_distance = des_r2.x2 - des_r1.x1 + 1;
    left = o2, right = o1;
  }

  if (cur_r1.y1 < cur_r2.y1) {
    v_distance = des_r1.y2 - des_r2.y1 + 1;
    top = o1, bottom = o2;
  } else {
    v_distance = des_r2.y2 - des_r1.y1 + 1;
    top = o2, bottom = o1;
  }

  /* Case analysis on overlap */
  if (h_overlap)
    move_away_h(left, right, h_distance);
  if (v_overlap)
    move_away_v(top, bottom, v_distance);
  if (!h_overlap && !v_overlap) {
    MDEBUG("Shoulder collision");
    if (h_distance >= v_distance)
      move_away_v(top, bottom, v_distance);
    else
      move_away_h(left, right, h_distance);
  }

  /* Assert that we've actually fixed the collision. */
  new_rect_from_object_des(&newdes_r1, o1);
  new_rect_from_object_des(&newdes_r2, o2);
  if (rects_overlap_h(&newdes_r1, &newdes_r2) &&
      rects_overlap_v(&newdes_r1, &newdes_r2)) {
    MDEBUG("assert: a collision was not well resolved!");
    object_dump(o1, "assert: ");
    object_dump(o2, "assert: ");
    rect_intersect_debuginfo(&des_r1, &des_r2);
    assert(!rects_overlap_h(&newdes_r1, &newdes_r2) ||
           !rects_overlap_v(&newdes_r1, &newdes_r2));
  }

  if (!IS_ACTIVE(o1) && !vectors_equal(o1->pos, o1->des_pos))
    object_make_active(m, o1);
  if (!IS_ACTIVE(o2) && !vectors_equal(o2->pos, o2->des_pos))
    object_make_active(m, o2);
}

/* Resolve collision player<->food */
static void
colr_player_food(Model *m, Model_object *player, Model_object *food) {
  if (IS_UNCONSCIOUS(player))
    return;

  if (player_gets_hit(m, player, food))
    return;
  player_pickup_food(m, player, food);
}

/* Resolve collision food<->home */
static void
colr_food_home(Model *m, Model_object *food, Model_object *home) {
  food_deliver(m, home, food);
}

/*****************************************************************************
 * Implementation of resolve_collisions                                      *
 *****************************************************************************/

/* Resolve collision between o1 and o2, calling func on them if they
 * collide.  Returns TRUE if both objects will have to be checked again against
 * all other objects in the other's list, FALSE otherwise. */
static int
resolve_object_pair(Model *m, Model_object *o1, Model_object *o2,
                    Collide_func f) {
  Rect des_r1, des_r2;

  if (IS_DELETED(o1) || IS_DELETED(o2))
    return FALSE;

  new_rect_from_object_des(&des_r1, o1);
  new_rect_from_object_des(&des_r2, o2);

  if (rects_collide(&des_r1, &des_r2)) {
    f(m, o1, o2);
  } else if (rects_overlap_v(&des_r1, &des_r2) && SGN(o1->pos->y - o2->pos->y)
             != SGN(o1->des_pos->y - o2->des_pos->y)) {
    MDEBUG("ghost: Vertical collision");
    object_dump(o1, "ghost: ");
    object_dump(o2, "ghost: ");
    f(m, o1, o2);
  } else if (rects_overlap_h(&des_r1, &des_r2) && SGN(o1->pos->x - o2->pos->x)
             != SGN(o1->des_pos->x - o2->des_pos->x)) {
    MDEBUG("ghost: Horizontal collision");
    object_dump(o1, "ghost: ");
    object_dump(o2, "ghost: ");
    f(m, o1, o2);
  } else if (SGN(o1->pos->x - o2->pos->x) !=
             SGN(o1->des_pos->x - o2->des_pos->x) &&
             SGN(o1->pos->y - o2->pos->y) !=
             SGN(o1->des_pos->y - o2->des_pos->y)) {
    MDEBUG("ghost: Diagonal collision");
    object_dump(o1, "ghost: ");
    object_dump(o2, "ghost: ");
    f(m, o1, o2);
  } else
    return FALSE;

  if (f == colr_move_away)
    return TRUE;
  else
    return FALSE;
}

/* Resolve collision between o1 and all objects in l, calling func on them if
 * they collide.  Return TRUE if everything has to be checked another time,
 * FALSE otherwise.  This function should only be used if o1 does not occur in
 * list l. */
static int
resolve_object_with_list(Model *m, Model_object *o1, List *l, Collide_func f) {
  List_ptr *ptr = new_list_ptr(l);
  Model_object *o2;

  if (list_ptr_first(ptr)) {
    do {
      o2 = (Model_object *)list_ptr_get_data(ptr);
      assert(o1 != o2);
      if (resolve_object_pair(m, o1, o2, f)) {
        del_list_ptr(ptr);
        return TRUE;
      }
    } while (list_ptr_next(ptr));
  }

  del_list_ptr(ptr);
  return FALSE;
}

/* Resolve collisions between all objects in l1 and those in l2, calling func
 * on them if they collide.  Return TRUE if everything has to be checked
 * another time, FALSE otherwise.  This function should only be used for two
 * separate lists; for resolving collisions between objects in one list use
 * resolve_within_list. */
static int
resolve_list_pair(Model *m, List *l1, List *l2, Collide_func f) {
  List_ptr *ptr = new_list_ptr(l1);

  if (list_ptr_first(ptr)) {
    do {
      if (resolve_object_with_list(m, list_ptr_get_data(ptr), l2, f)) {
        del_list_ptr(ptr);
        return TRUE;
      }
    } while (list_ptr_next(ptr));
  }

  del_list_ptr(ptr);
  return FALSE;
}

/* Resolve collisions between all pairs of objects in list l, calling func on
 * them if they collide.  Return TRUE if everything has to be checked another
 * time, FALSE otherwise. */
static int
resolve_within_list(Model *m, List *l, Collide_func f) {
  List_ptr *ptr1 = new_list_ptr(l), *ptr2;
  Model_object *o1, *o2;

  if (list_ptr_first(ptr1)) {
    do {
      ptr2 = list_ptr_dup(ptr1);
      while (list_ptr_next(ptr2)) {
        o1 = (Model_object *)list_ptr_get_data(ptr1);
        o2 = (Model_object *)list_ptr_get_data(ptr2);
        assert(o1 != o2);
        if (resolve_object_pair(m, o1, o2, f)) {
          del_list_ptr(ptr2);
          del_list_ptr(ptr1);
          return TRUE;
        }
      }
      del_list_ptr(ptr2);
    } while (list_ptr_next(ptr1));
  }

  del_list_ptr(ptr1);
  return FALSE;
}

/* Macros for resolve_collisions */
#define RESOLVE_ONE(c, l) \
  if ((check & (c)) && resolve_within_list(m, l, colr_move_away)) \
    recheck |= (c)
#define RESOLVE_TWO(c1, l1, c2, l2) \
  if ((check & (c1) || check & (c2)) && \
      resolve_list_pair(m, l1, l2, colr_move_away)) \
    recheck |= (c1) | (c2)

/* Resolve collisions all collisions in m in a useful way. */
void
resolve_collisions(Model *m) {
  int check = ~0, recheck;
  MDEBUG("Starting collision resolving");
  
  /* Move objects away from each other, multiple times if required. */
  do {
    recheck = 0;

    /* Resolve collisions in one list */
    RESOLVE_ONE(RE_FOOD,     m->act_food);
    RESOLVE_ONE(RE_PLAYERS,  m->act_players);

    /* Resolve collisions in two lists */
    RESOLVE_TWO(RE_UPLAYERS, m->act_uplayers,
                RE_BOXES,    m->boxes);
    RESOLVE_TWO(RE_FOOD,     m->act_food,
                RE_BOXES,    m->boxes);
    RESOLVE_TWO(RE_BOXES,    m->boxes,
                RE_PLAYERS,  m->act_players);
    RESOLVE_TWO(RE_FOOD,     m->act_food,
                RE_NOCHECK,  m->pas_food);
    RESOLVE_TWO(RE_PLAYERS,  m->act_players,
                RE_NOCHECK,  m->pas_players);

    check = recheck;
  } while (recheck);

  /* Do object type-specific collision resolving. */
  resolve_list_pair(m, m->act_food,    m->homes,    colr_food_home);
  resolve_list_pair(m, m->act_players, m->act_food, colr_player_food);
  resolve_list_pair(m, m->pas_players, m->act_food, colr_player_food);
  resolve_list_pair(m, m->act_players, m->pas_food, colr_player_food);
  resolve_list_pair(m, m->pas_players, m->pas_food, colr_player_food);

  MDEBUG("Done with collision resolving");
}
