/*
 * gldgStroke.c
 *
 * OpenGL Graph Display stroke font module implementation
 *
 * Written by: Shawn Taras
 */
#include <stdlib.h>
#include <math.h>
#include <GL/gl.h>
#include "glgd.h"

/*
 * Static local (to this module) variables
 */
static glgdStroke       *s_currentStroke = NULL;
static GLint            s_blendFunc[2] = {GL_ONE, GL_ZERO};

/* ASCII Character tristrips drawn into a 16x16 pixel area
 * --> NOTE! origin of 16x16 area is lower-left corner
 */
static const unsigned char s_g20[] =
{
    0xff, 0xff
};
static const unsigned char s_g21[] =
{
    0x60, 0x80, 0x62, 0x82, 0xff, 0x64, 0x84, 0x6e,
    0x8e, 0xff, 0xff
};
static const unsigned char s_g22[] =
{
    0x6a, 0x6e, 0x4c, 0x4e, 0xff, 0x8a, 0xac, 0x8e,
    0xae, 0xff, 0xff
};
static const unsigned char s_g23[] =
{
    0x42, 0x62, 0x44, 0x64, 0x46, 0x66, 0x48, 0x68,
    0x4a, 0x6a, 0x4c, 0x6c, 0xff, 0x82, 0xa2, 0x84,
    0xa4, 0x86, 0xa6, 0x88, 0xa8, 0x8a, 0xaa, 0x8c,
    0xac, 0xff, 0x24, 0x44, 0x26, 0x46, 0xff, 0x64,
    0x84, 0x66, 0x86, 0xff, 0xa4, 0xc4, 0xa6, 0xc6,
    0xff, 0x28, 0x48, 0x2a, 0x4a, 0xff, 0x68, 0x88,
    0x6a, 0x8a, 0xff, 0xa8, 0xc8, 0xaa, 0xca, 0xff,
    0xff
};
static const unsigned char s_g24[] =
{
    0x46, 0x48, 0x26, 0x38, 0x08, 0x29, 0x0a, 0x3a,
    0x2c, 0x4a, 0x4c, 0xff, 0xa8, 0xa6, 0xc8, 0xb6,
    0xe6, 0xc5, 0xe4, 0xb4, 0xc2, 0xa4, 0xa2, 0xff,
    0x25, 0x04, 0x34, 0x22, 0x44, 0x42, 0xff, 0xc9,
    0xea, 0xba, 0xcc, 0xaa, 0xac, 0xff, 0x40, 0x60,
    0x42, 0x62, 0x44, 0x64, 0x46, 0x66, 0x48, 0x68,
    0x4a, 0x6a, 0x4c, 0x6c, 0x4e, 0x6e, 0xff, 0x80,
    0xa0, 0x82, 0xa2, 0x84, 0xa4, 0x86, 0xa6, 0x88,
    0xa8, 0x8a, 0xaa, 0x8c, 0xac, 0x8e, 0xae, 0xff,
    0x62, 0x82, 0x64, 0x84, 0xff, 0x66, 0x86, 0x68,
    0x88, 0xff, 0x6a, 0x8a, 0x6c, 0x8c, 0xff, 0xff
};
static const unsigned char s_g25[] =
{
    0x00, 0x20, 0x02, 0xec, 0xce, 0xee, 0xff, 0xa0,
    0xb2, 0x82, 0xa3, 0x84, 0xb4, 0xa6, 0xc6, 0xff,
    0xc6, 0xb4, 0xe4, 0xc3, 0xe2, 0xb2, 0xc0, 0xa0,
    0xff, 0x28, 0x3a, 0x0a, 0x2b, 0x0c, 0x3c, 0x2e,
    0x4e, 0xff, 0x4e, 0x3c, 0x6c, 0x4b, 0x6a, 0x3a,
    0x48, 0x28, 0xff, 0xff
};
static const unsigned char s_g26[] =
{
    0xa0, 0xe0, 0x91, 0xb3, 0x73, 0x95, 0x55, 0x77,
    0x37, 0x59, 0x28, 0x4a, 0x2c, 0x4b, 0x4e, 0x5c,
    0x6e, 0x6b, 0x8c, 0x6a, 0x88, 0x59, 0x77, 0xff,
    0x91, 0x73, 0x80, 0x62, 0x20, 0x42, 0x02, 0x33,
    0x04, 0x55, 0x37, 0xff, 0x95, 0xb3, 0xa6, 0xc4,
    0xff, 0xff
};
static const unsigned char s_g27[] =
{
    0x6a, 0x8c, 0x6e, 0x8e, 0xff, 0xff
};
static const unsigned char s_g28[] =
{
    0x80, 0xa0, 0x62, 0x82, 0x6c, 0x8c, 0x8e, 0xae,
    0xff, 0xff
};
static const unsigned char s_g29[] =
{
    0x40, 0x60, 0x62, 0x82, 0x6c, 0x8c, 0x4e, 0x6e,
    0xff, 0xff
};
static const unsigned char s_g2A[] =
{
    0x77, 0xa8, 0x8a, 0xb9, 0x9b, 0xff, 0x77, 0x8a,
    0x6a, 0x8c, 0x6c, 0xff, 0x77, 0x6a, 0x48, 0x5b,
    0x39, 0xff, 0x77, 0x48, 0x46, 0x28, 0x26, 0xff,
    0x77, 0x46, 0x64, 0x35, 0x53, 0xff, 0x77, 0x64,
    0x84, 0x62, 0x82, 0xff, 0x77, 0x84, 0xa6, 0x93,
    0xb5, 0xff, 0x77, 0xa6, 0xa8, 0xc6, 0xc8, 0xff,
    0xff
};
static const unsigned char s_g2B[] =
{
    0x62, 0x82, 0x66, 0x86, 0x88, 0xc6, 0xc8, 0xff,
    0x8c, 0x6c, 0x88, 0x68, 0x66, 0x28, 0x26, 0xff,
    0xff
};
static const unsigned char s_g2C[] =
{
    0x84, 0x64, 0x82, 0x60, 0xff, 0xff
};
static const unsigned char s_g2D[] =
{
    0x28, 0x26, 0xc8, 0xc6, 0xff, 0xff
};
static const unsigned char s_g2E[] =
{
    0x60, 0x80, 0x62, 0x82, 0xff, 0xff
};
static const unsigned char s_g2F[] =
{
    0x00, 0x20, 0x02, 0xec, 0xce, 0xee, 0xff, 0xff
};
static const unsigned char s_g30[] =
{
    0x24, 0x42, 0xac, 0xca, 0xff, 0xb2, 0xc0, 0xc3,
    0xe2, 0xca, 0xec, 0xac, 0xce, 0x3c, 0x2e, 0x2b,
    0x0c, 0x24, 0x02, 0x42, 0x20, 0xb2, 0xc0, 0xff,
    0xff
};
static const unsigned char s_g31[] =
{
    0x80, 0xff, 0x4c, 0x4a, 0x6e, 0x6a, 0x8e, 0x60,
    0x80, 0xff, 0xff
};
static const unsigned char s_g32[] =
{
    0x0c, 0x2b, 0x2e, 0x3c, 0xce, 0xbc, 0xec, 0xcb,
    0xe8, 0xc9, 0xc6, 0xb8, 0x36, 0x28, 0x25, 0x06,
    0x22, 0x00, 0xe2, 0xe0, 0xff, 0xff
};
static const unsigned char s_g33[] =
{
    0x0c, 0x2b, 0x2e, 0x3c, 0xce, 0xbc, 0xec, 0xcb,
    0xe8, 0xc9, 0xd7, 0xb8, 0xb6, 0x68, 0x66, 0xff,
    0xd7, 0xb6, 0xe6, 0xc5, 0xe2, 0xc3, 0xc0, 0xb2,
    0x20, 0x32, 0x02, 0x23, 0xff, 0xff
};
static const unsigned char s_g34[] =
{
    0xc6, 0xc8, 0x06, 0x28, 0x0c, 0x2c, 0xff, 0xc0,
    0xe0, 0xc6, 0xe7, 0xc8, 0xee, 0xce, 0xff, 0xff
};
static const unsigned char s_g35[] =
{
    0xec, 0xee, 0x2c, 0x0e, 0x29, 0x08, 0x38, 0x26,
    0xc8, 0xb6, 0xe6, 0xc5, 0xe2, 0xc3, 0xc0, 0xb2,
    0x20, 0x32, 0x02, 0x23, 0xff, 0xff
};
static const unsigned char s_g36[] =
{
    0xcb, 0xec, 0xbc, 0xce, 0x3c, 0x2e, 0x2b, 0x0c,
    0x28, 0x07, 0x26, 0x02, 0x23, 0x20, 0x32, 0xc0,
    0xb2, 0xe2, 0xc3, 0xe6, 0xc5, 0xc8, 0xb6, 0x28,
    0x26, 0xff, 0xff
};
static const unsigned char s_g37[] =
{
    0x0e, 0x0c, 0xee, 0xbc, 0xec, 0x67, 0x86, 0x60,
    0x80, 0xff, 0xff
};
static const unsigned char s_g38[] =
{
    0x36, 0x17, 0x25, 0x06, 0x23, 0x02, 0x32, 0x20,
    0xb2, 0xc0, 0xc3, 0xe2, 0xc5, 0xe6, 0xb6, 0xd7,
    0xb8, 0xe8, 0xc9, 0xec, 0xcb, 0xce, 0xbc, 0x2e,
    0x3c, 0x0c, 0x2b, 0x08, 0x29, 0x17, 0x38, 0x36,
    0xb8, 0xb6, 0xff, 0xff
};
static const unsigned char s_g39[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xc6, 0xe7, 0xc8, 0xec, 0xcb, 0xce, 0xbc, 0x2e,
    0x3c, 0x0c, 0x2b, 0x08, 0x29, 0x26, 0x38, 0xc6,
    0xc8, 0xff, 0xff
};
static const unsigned char s_g3A[] =
{
    0x60, 0x80, 0x62, 0x82, 0xff, 0x66, 0x86, 0x68,
    0x88, 0xff, 0xff
};
static const unsigned char s_g3B[] =
{
    0x84, 0x64, 0x82, 0x60, 0xff, 0x66, 0x86, 0x68,
    0x88, 0xff, 0xff
};
static const unsigned char s_g3C[] =
{
    0xbc, 0x8c, 0x67, 0x37, 0xb2, 0x82, 0xff, 0xff
};
static const unsigned char s_g3D[] =
{
    0x24, 0x26, 0xc4, 0xc6, 0xff, 0x28, 0x2a, 0xc8,
    0xca, 0xff, 0xff
};
static const unsigned char s_g3E[] =
{
    0x32, 0x62, 0x87, 0xb7, 0x3c, 0x6c, 0xff, 0xff
};
static const unsigned char s_g3F[] =
{
    0x60, 0x80, 0x62, 0x82, 0xff, 0x64, 0x84, 0x65,
    0xea, 0xcb, 0xec, 0xbc, 0xce, 0x3c, 0x2e, 0x2b,
    0x0c, 0xff, 0xff
};
static const unsigned char s_g40[] =
{
    0xe2, 0xc3, 0xc0, 0xb2, 0x20, 0x32, 0x02, 0x23,
    0x0c, 0x2b, 0x2e, 0x3c, 0xce, 0xbc, 0xec, 0xcb,
    0xe6, 0xc7, 0xc4, 0xb6, 0xa4, 0xa6, 0x84, 0x86,
    0x64, 0x76, 0x46, 0x67, 0x48, 0x78, 0x6a, 0x88,
    0xaa, 0x86, 0xa6, 0xff, 0xff
};
static const unsigned char s_g41[] =
{
    0x00, 0x20, 0x08, 0x26, 0x38, 0xc6, 0xb8, 0xff,
    0xc0, 0xe0, 0xc6, 0xe8, 0xb8, 0x8e, 0x7c, 0x6e,
    0x38, 0x08, 0xff, 0xff
};
static const unsigned char s_g42[] =
{
    0xb8, 0xd7, 0xc9, 0xe8, 0xcb, 0xec, 0xbc, 0xce,
    0x2c, 0x0e, 0x28, 0x07, 0x26, 0x00, 0x22, 0xc0,
    0xb2, 0xe2, 0xc3, 0xe6, 0xc5, 0xd7, 0xb6, 0xb8,
    0x26, 0x28, 0xff, 0xff
};
static const unsigned char s_g43[] =
{
    0xcb, 0xec, 0xbc, 0xce, 0x3c, 0x2e, 0x2b, 0x0c,
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xff, 0xff
};
static const unsigned char s_g44[] =
{
    0x22, 0x00, 0xb2, 0xc0, 0xc3, 0xe2, 0xcb, 0xec,
    0xbc, 0xce, 0x2c, 0x0e, 0x22, 0x00, 0xff, 0xff
};
static const unsigned char s_g45[] =
{
    0xe0, 0xe2, 0x00, 0x22, 0x06, 0x26, 0x08, 0x28,
    0x0e, 0x2c, 0xee, 0xec, 0xff, 0x28, 0x26, 0x88,
    0x86, 0xff, 0xff
};
static const unsigned char s_g46[] =
{
    0x00, 0x20, 0x06, 0x26, 0x08, 0x28, 0x0e, 0x2c,
    0xee, 0xec, 0xff, 0x28, 0x26, 0x88, 0x86, 0xff,
    0xff
};
static const unsigned char s_g47[] =
{
    0xcb, 0xec, 0xbc, 0xce, 0x3c, 0x2e, 0x2b, 0x0c,
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xc6, 0xe8, 0x86, 0x88, 0xff, 0xff
};
static const unsigned char s_g48[] =
{
    0x2e, 0x0e, 0x28, 0x07, 0x26, 0x00, 0x20, 0xff,
    0xc0, 0xe0, 0xc6, 0xe7, 0xc8, 0xee, 0xce, 0xff,
    0x28, 0x26, 0xc8, 0xc6, 0xff, 0xff
};
static const unsigned char s_g49[] =
{
    0x42, 0x40, 0x62, 0x60, 0x82, 0x80, 0xa2, 0xa0,
    0xff, 0x4e, 0x4c, 0x6e, 0x6c, 0x8e, 0x8c, 0xae,
    0xac, 0xff, 0x62, 0x82, 0x6c, 0x8c, 0xff, 0xff
};
static const unsigned char s_g4A[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xce, 0xee, 0xff, 0xff
};
static const unsigned char s_g4B[] =
{
    0x2e, 0x0e, 0x28, 0x07, 0x26, 0x00, 0x20, 0xff,
    0x28, 0x26, 0x68, 0x66, 0x97, 0xc0, 0xe2, 0xe0,
    0xff, 0x68, 0x97, 0xce, 0xec, 0xee, 0xff, 0xff
};
static const unsigned char s_g4C[] =
{
    0xe0, 0xe2, 0x00, 0x22, 0x0e, 0x2e, 0xff, 0xff
};
static const unsigned char s_g4D[] =
{
    0x00, 0x20, 0x0e, 0x2b, 0x2e, 0x67, 0x79, 0x87,
    0xce, 0xcb, 0xee, 0xc0, 0xe0, 0xff, 0xff
};
static const unsigned char s_g4E[] =
{
    0x00, 0x20, 0x0e, 0x2b, 0x2e, 0xc0, 0xc3, 0xe0,
    0xce, 0xee, 0xff, 0xff
};
static const unsigned char s_g4F[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xcb, 0xec, 0xbc, 0xce, 0x3c, 0x2e, 0x2b, 0x0c,
    0x23, 0x02, 0xff, 0xff
};
static const unsigned char s_g50[] =
{
    0x00, 0x20, 0x06, 0x26, 0x08, 0x28, 0x0c, 0x2c,
    0x0e, 0xbc, 0xce, 0xcb, 0xec, 0xc9, 0xe8, 0xb8,
    0xc6, 0x28, 0x26, 0xff, 0xff
};
static const unsigned char s_g51[] =
{
    0xc0, 0xe2, 0xc4, 0xec, 0xcb, 0xce, 0xbc, 0x2e,
    0x3c, 0x0c, 0x2b, 0x02, 0x23, 0x20, 0x32, 0xc0,
    0xa2, 0xc4, 0x84, 0xa6, 0xff, 0xff
};
static const unsigned char s_g52[] =
{
    0xc0, 0xe0, 0xc5, 0xe6, 0xb6, 0xd7, 0xb8, 0xe8,
    0xc9, 0xec, 0xcb, 0xce, 0xbc, 0x0e, 0x2c, 0x08,
    0x28, 0x06, 0x26, 0x00, 0x20, 0xff, 0x28, 0x26,
    0xb8, 0xb6, 0xff, 0xff
};
static const unsigned char s_g53[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xc5, 0xe6, 0xb6, 0xc8, 0x26, 0x38, 0x08, 0x29,
    0x0c, 0x2b, 0x2e, 0x3c, 0xce, 0xbc, 0xec, 0xcb,
    0xff, 0xff
};
static const unsigned char s_g54[] =
{
    0x0e, 0x0c, 0x6e, 0x6c, 0x8e, 0x8c, 0xee, 0xec,
    0xff, 0x60, 0x80, 0x6c, 0x8c, 0xff, 0xff
};
static const unsigned char s_g55[] =
{
    0x2e, 0x0e, 0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0,
    0xc3, 0xe2, 0xce, 0xee, 0xff, 0xff
};
static const unsigned char s_g56[] =
{
    0x2e, 0x0e, 0x27, 0x06, 0x72, 0x60, 0xff, 0x60,
    0x80, 0x72, 0xe6, 0xc7, 0xee, 0xce, 0xff, 0xff
};
static const unsigned char s_g57[] =
{
    0xee, 0xce, 0xe0, 0xc3, 0xc0, 0x87, 0x75, 0x67,
    0x20, 0x23, 0x00, 0x2e, 0x0e, 0xff, 0xff
};
static const unsigned char s_g58[] =
{
    0x00, 0x20, 0x02, 0x75, 0x57, 0x97, 0x79, 0xec,
    0xce, 0xee, 0xff, 0xe0, 0xe2, 0xc0, 0x97, 0x75,
    0xff, 0x0e, 0x0c, 0x2e, 0x57, 0x79, 0xff, 0xff
};
static const unsigned char s_g59[] =
{
    0x0e, 0x0c, 0x2e, 0x66, 0x79, 0x86, 0xce, 0xec,
    0xee, 0xff, 0x60, 0x80, 0x66, 0x86, 0xff, 0xff
};
static const unsigned char s_g5A[] =
{
    0x0e, 0x0c, 0xee, 0xbc, 0xec, 0x02, 0x32, 0x00,
    0xe2, 0xe0, 0xff, 0xff
};
static const unsigned char s_g5B[] =
{
    0xa0, 0xa2, 0x60, 0x82, 0x6e, 0x8c, 0xae, 0xac,
    0xff, 0xff
};
static const unsigned char s_g5C[] =
{
    0xe0, 0xe2, 0xc0, 0x2e, 0x0c, 0x0e, 0xff, 0xff
};
static const unsigned char s_g5D[] =
{
    0x4e, 0x4c, 0x8e, 0x6c, 0x80, 0x62, 0x40, 0x42,
    0xff, 0xff
};
static const unsigned char s_g5E[] =
{
    0x3a, 0x6a, 0x7e, 0x7b, 0xba, 0x8a, 0xff, 0xff
};
static const unsigned char s_g5F[] =
{
    0x02, 0x00, 0xe2, 0xe0, 0xff, 0xff
};
static const unsigned char s_g60[] =
{
    0x8a, 0x8e, 0x6c, 0x6e, 0xff, 0xff
};
static const unsigned char s_g61[] =
{
    0x08, 0x27, 0x2a, 0x38, 0xca, 0xb8, 0xe8, 0xc7,
    0xe6, 0xc6, 0xe4, 0xc4, 0xe0, 0xc2, 0x20, 0x32,
    0x02, 0x23, 0x04, 0x34, 0x26, 0xc4, 0xc6, 0xff,
    0xff
};
static const unsigned char s_g62[] =
{
    0x2e, 0x0e, 0x2a, 0x09, 0x28, 0x00, 0x22, 0xc0,
    0xb2, 0xe2, 0xc3, 0xe8, 0xc7, 0xca, 0xb8, 0x2a,
    0x28, 0xff, 0xff
};
static const unsigned char s_g63[] =
{
    0xc7, 0xe8, 0xb8, 0xca, 0x38, 0x2a, 0x27, 0x08,
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xff, 0xff
};
static const unsigned char s_g64[] =
{
    0xc8, 0xca, 0x38, 0x2a, 0x27, 0x08, 0x23, 0x02,
    0x32, 0x20, 0xc2, 0xe0, 0xc8, 0xe9, 0xca, 0xee,
    0xce, 0xff, 0xff
};
static const unsigned char s_g65[] =
{
    0xe2, 0xc3, 0xc0, 0xb2, 0x20, 0x32, 0x02, 0x23,
    0x04, 0x24, 0x06, 0x26, 0x08, 0x27, 0x2a, 0x38,
    0xca, 0xb8, 0xe8, 0xc7, 0xe4, 0xc6, 0x24, 0x26,
    0xff, 0xff
};
static const unsigned char s_g66[] =
{
    0x4a, 0x48, 0x6a, 0x68, 0x8a, 0x88, 0xaa, 0xa8,
    0xff, 0x60, 0x80, 0x68, 0x88, 0xff, 0x6a, 0x8a,
    0x6c, 0x8b, 0x8e, 0x9c, 0xae, 0xac, 0xff, 0xff
};
static const unsigned char s_g67[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xc4, 0xe6, 0xc6, 0xea, 0xc8, 0x2a, 0x38, 0x08,
    0x27, 0x06, 0x36, 0x24, 0xc6, 0xc4, 0xff, 0xff
};
static const unsigned char s_g68[] =
{
    0x00, 0x20, 0x08, 0x28, 0x0a, 0x2a, 0x0e, 0x2e,
    0xff, 0x2a, 0x28, 0xca, 0xb8, 0xe8, 0xc7, 0xe0,
    0xc0, 0xff, 0xff
};
static const unsigned char s_g69[] =
{
    0x60, 0x80, 0x6a, 0x8a, 0xff, 0x6c, 0x8c, 0x6e,
    0x8e, 0xff, 0xff
};
static const unsigned char s_g6A[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xca, 0xea, 0xff, 0xcc, 0xec, 0xce, 0xee, 0xff,
    0xff
};
static const unsigned char s_g6B[] =
{
    0x00, 0x20, 0x04, 0x24, 0x06, 0x26, 0x0e, 0x2e,
    0xff, 0x26, 0x24, 0x86, 0x84, 0xb5, 0xc0, 0xe2,
    0xe0, 0xff, 0x86, 0xb5, 0xca, 0xe8, 0xea, 0xff,
    0xff
};
static const unsigned char s_g6C[] =
{
    0x60, 0x80, 0x6e, 0x8e, 0xff, 0xff
};
static const unsigned char s_g6D[] =
{
    0x00, 0x20, 0x0a, 0x28, 0x6a, 0x68, 0xff, 0x60,
    0x80, 0x68, 0x88, 0x6a, 0xb8, 0xca, 0xc7, 0xe8,
    0xc0, 0xe0, 0xff, 0xff
};
static const unsigned char s_g6E[] =
{
    0x00, 0x20, 0x0a, 0x28, 0xca, 0xb8, 0xe8, 0xc7,
    0xe0, 0xc0, 0xff, 0xff
};
static const unsigned char s_g6F[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xc7, 0xe8, 0xb8, 0xca, 0x38, 0x2a, 0x27, 0x08,
    0x23, 0x02, 0xff, 0xff
};
static const unsigned char s_g70[] =
{
    0x00, 0x20, 0x04, 0x24, 0x06, 0x26, 0x0a, 0x28,
    0xca, 0xb8, 0xe8, 0xc7, 0xe6, 0xb6, 0xc4, 0x26,
    0x24, 0xff, 0xff
};
static const unsigned char s_g71[] =
{
    0xc0, 0xe0, 0xc4, 0xe5, 0xc6, 0xea, 0xc8, 0x2a,
    0x38, 0x08, 0x27, 0x06, 0x36, 0x24, 0xc6, 0xc4,
    0xff, 0xff
};
static const unsigned char s_g72[] =
{
    0x00, 0x20, 0x0a, 0x28, 0xca, 0xb8, 0xe8, 0xc7,
    0xff, 0xff
};
static const unsigned char s_g73[] =
{
    0xc3, 0xe4, 0xb4, 0xc6, 0x24, 0x36, 0x06, 0x27,
    0x08, 0x38, 0x2a, 0xb8, 0xca, 0xc7, 0xe8, 0xff,
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xe4, 0xff, 0xff
};
static const unsigned char s_g74[] =
{
    0x4a, 0x48, 0x6a, 0x68, 0xff, 0x8a, 0x88, 0xaa,
    0xa8, 0xff, 0xa0, 0xa2, 0x80, 0x92, 0x62, 0x83,
    0x68, 0x88, 0x6a, 0x8a, 0x6c, 0x8c, 0xff, 0xff
};
static const unsigned char s_g75[] =
{
    0x2a, 0x0a, 0x23, 0x02, 0x32, 0x20, 0xc2, 0xe0,
    0xca, 0xea, 0xff, 0xff
};
static const unsigned char s_g76[] =
{
    0xea, 0xca, 0xe6, 0xc7, 0x80, 0x72, 0x60, 0x27,
    0x06, 0x2a, 0x0a, 0xff, 0xff
};
static const unsigned char s_g77[] =
{
    0x2a, 0x0a, 0x22, 0x00, 0x52, 0x60, 0x63, 0x71,
    0x83, 0x80, 0x92, 0xc0, 0xb2, 0xe2, 0xc3, 0xea,
    0xca, 0xff, 0x8a, 0x6a, 0x83, 0x63, 0xff, 0xff
};
static const unsigned char s_g78[] =
{
    0x00, 0x20, 0x02, 0x74, 0x45, 0x76, 0x08, 0x2a,
    0x0a, 0xff, 0xea, 0xca, 0xe8, 0x76, 0xa5, 0x74,
    0xe2, 0xc0, 0xe0, 0xff, 0xff
};
static const unsigned char s_g79[] =
{
    0x23, 0x02, 0x32, 0x20, 0xb2, 0xc0, 0xc3, 0xe2,
    0xc4, 0xe5, 0xc6, 0xea, 0xca, 0xff, 0x2a, 0x0a,
    0x27, 0x06, 0x36, 0x24, 0xc6, 0xc4, 0xff, 0xff
};
static const unsigned char s_g7A[] =
{
    0x0a, 0x08, 0xea, 0xa8, 0xe8, 0x02, 0x42, 0x00,
    0xe2, 0xe0, 0xff, 0xff
};
static const unsigned char s_g7B[] =
{
    0x80, 0xa0, 0x62, 0x82, 0x66, 0x86, 0x57, 0x77,
    0x68, 0x88, 0x6c, 0x8c, 0x8e, 0xae, 0xff, 0xff
};
static const unsigned char s_g7C[] =
{
    0x60, 0x80, 0x6f, 0x8f, 0xff, 0xff
};
static const unsigned char s_g7D[] =
{
    0x6e, 0x4e, 0x8c, 0x6c, 0x88, 0x68, 0x97, 0x77,
    0x86, 0x66, 0x82, 0x62, 0x60, 0x40, 0xff, 0xff
};
static const unsigned char s_g7E[] =
{
    0xbc, 0xbe, 0x9a, 0x9c, 0x5c, 0x5e, 0x3a, 0x3c,
    0xff, 0xff
};
static const unsigned char s_g7F[] =
{
    0x44, 0x22, 0xa4, 0xc2, 0xaa, 0xcc, 0x4a, 0x2c,
    0x44, 0x22, 0xff, 0xff
};
static const unsigned char *s_stroke[] =
{
    s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F,
    s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F,
    s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F,
    s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F, s_g7F,
    s_g20, s_g21, s_g22, s_g23, s_g24, s_g25, s_g26, s_g27,
    s_g28, s_g29, s_g2A, s_g2B, s_g2C, s_g2D, s_g2E, s_g2F,
    s_g30, s_g31, s_g32, s_g33, s_g34, s_g35, s_g36, s_g37,
    s_g38, s_g39, s_g3A, s_g3B, s_g3C, s_g3D, s_g3E, s_g3F,
    s_g40, s_g41, s_g42, s_g43, s_g44, s_g45, s_g46, s_g47,
    s_g48, s_g49, s_g4A, s_g4B, s_g4C, s_g4D, s_g4E, s_g4F,
    s_g50, s_g51, s_g52, s_g53, s_g54, s_g55, s_g56, s_g57,
    s_g58, s_g59, s_g5A, s_g5B, s_g5C, s_g5D, s_g5E, s_g5F,
    s_g60, s_g61, s_g62, s_g63, s_g64, s_g65, s_g66, s_g67,
    s_g68, s_g69, s_g6A, s_g6B, s_g6C, s_g6D, s_g6E, s_g6F,
    s_g70, s_g71, s_g72, s_g73, s_g74, s_g75, s_g76, s_g77,
    s_g78, s_g79, s_g7A, s_g7B, s_g7C, s_g7D, s_g7E, s_g7F
};

/*
 * Static local (to this module) functions
 */
static void
glgdStrokePushAttributes(void)
{
    glPushAttrib(GL_ENABLE_BIT);
    glGetIntegerv(GL_BLEND_SRC, &s_blendFunc[0]);
    glGetIntegerv(GL_BLEND_DST, &s_blendFunc[1]);
    
    /* Common attributes for primitive drawing */
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
}

static void
glgdStrokePopAttributes(void)
{
    glPopAttrib();
    glBlendFunc(s_blendFunc[0], s_blendFunc[1]);
}

/*
 * External public functions
 */
glgdStroke
*glgdStrokeCreate(void)
{
    glgdStroke  *stroke;
    
    stroke = (glgdStroke *)GLGD_MALLOC(sizeof(glgdStroke));
    if (stroke)
    {
        glgdStrokeInit(stroke);
    }
    
    return stroke;
}

glgdStroke
*glgdStrokeDestroy(glgdStroke *stroke)
{
    if (stroke)
    {
        GLGD_FREE(stroke);
    }
    
    return (glgdStroke *)NULL;
}

void
glgdStrokeInit(glgdStroke *stroke)
{
    if (stroke)
    {
        stroke->flags = GLGDSTROKE_FLAG_INITIALIZED;
        stroke->tabSize = GLGDSTROKE_TABSIZE_DEFAULT;
        stroke->windowDim[0] = 640.0;
        stroke->windowDim[1] = 480.0;
        stroke->pointSize[0] = GLGDSTROKE_POINTSIZEX_DEFAULT;
        stroke->pointSize[1] = GLGDSTROKE_POINTSIZEY_DEFAULT;
        stroke->pos[0] = 0.0;
        stroke->pos[1] = 0.0;
        glgdStrokeClipFullWindow(stroke);
        stroke->col[0] = 1.0;
        stroke->col[1] = 1.0;
        stroke->col[2] = 1.0;
        stroke->col[3] = 1.0;
    }
}

void
glgdStrokeFini(glgdStroke *stroke)
{
    glgdStrokeInit(stroke);
}

glgdStroke
*glgdStrokeGetCurrent(void)
{
    return s_currentStroke;
}

void
glgdStrokeSetCurrent(glgdStroke *stroke)
{
    s_currentStroke = stroke;
}

int
glgdStrokePrint(glgdStroke *stroke, const char *fmt, ...)
{
    int         rc;
    va_list     args;

    va_start(args, fmt);
    rc = glgdStrokePrintVar(fmt, args);
    va_end(args);

    return rc;
}

int
glgdStrokePrintVar(const char *fmt, va_list ap)
{
    int         rc;
    int         vtxNdx;
    char        str[256];
    char        *c;
    glgdStroke  *stroke;

    rc = 0;
    stroke = glgdStrokeGetCurrent();
    if (stroke)
    {
        /* Restrict string to 255 characters */
        rc = vsnprintf(str, 255, fmt, ap);

        /* Initialize the rendering */
        glgdStrokePushAttributes();
        glBegin(GL_TRIANGLE_STRIP);
        
        /* Set the text color */
        if (stroke->col[3] < 1.0)
        {
            glEnable(GL_BLEND);
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glColor4d(stroke->col[0], stroke->col[1],
                stroke->col[2], stroke->col[3]);
        }
        else
        {
            glDisable(GL_BLEND);
            glColor3d(stroke->col[0], stroke->col[1], stroke->col[2]);
        }
        
        /* Submit the vertices for the string */
        vtxNdx = 0;
        c = &str[0];
        while (*c)
        {
            vtxNdx = glgdStrokeBuild(stroke, *c, vtxNdx);

            c++;
        }
        
        glEnd();
        glgdStrokePopAttributes();
    }
    
    return rc;
}

void
glgdStrokeTabSizeSet(glgdStroke *stroke, int tabSize)
{
    if (stroke)
    {
        if (tabSize < 0)
        {
            stroke->tabSize = GLGDSTROKE_TABSIZE_DEFAULT;
        }
        else
        {
            stroke->tabSize = tabSize;
        }
    }
}

void
glgdStrokePointSizeSet(glgdStroke *stroke, glgdVec2 pointSize)
{
    if (stroke)
    {
        if (pointSize[0] < 0.0 || pointSize[1] < 0.0)
        {
            stroke->pointSize[0] = GLGDSTROKE_POINTSIZEX_DEFAULT;
            stroke->pointSize[1] = GLGDSTROKE_POINTSIZEY_DEFAULT;
        }
        else
        {
            stroke->pointSize[0] = pointSize[0];
            stroke->pointSize[1] = pointSize[1];
        }
    }
}

void
glgdStrokePointSizeSetByList
(
    glgdStroke  *stroke,
    GLdouble    w,
    GLdouble    h
)
{
    if (stroke)
    {
        if (w < 0.0 || h < 0.0)
        {
            stroke->pointSize[0] = GLGDSTROKE_POINTSIZEX_DEFAULT;
            stroke->pointSize[1] = GLGDSTROKE_POINTSIZEY_DEFAULT;
        }
        else
        {
            stroke->pointSize[0] = w;
            stroke->pointSize[1] = h;
        }
    }
}

void
glgdStrokePosSet(glgdStroke *stroke, glgdVec2 pos)
{
    if (stroke)
    {
        stroke->pos[0] = pos[0];
        stroke->pos[1] = pos[1];
    }
}

void
glgdStrokePosSetByList(glgdStroke *stroke, GLdouble x, GLdouble y)
{
    if (stroke)
    {
        stroke->pos[0] = x;
        stroke->pos[1] = y;
    }
}

void
glgdStrokeWindowDimSet(glgdStroke *stroke, glgdVec2 windowDim)
{
    if (stroke)
    {
        stroke->windowDim[0] = windowDim[0];
        stroke->windowDim[1] = windowDim[1];
    }
}

void
glgdStrokeWindowDimSetByList(glgdStroke *stroke, GLdouble w, GLdouble h)
{
    if (stroke)
    {
        stroke->windowDim[0] = w;
        stroke->windowDim[1] = h;
    }
}

void
glgdStrokeClipFullWindow(glgdStroke *stroke)
{
    if (stroke)
    {
        stroke->clip[0] = 0.0;
        stroke->clip[1] = 0.0;
        stroke->clip[2] = stroke->windowDim[0];
        stroke->clip[3] = stroke->windowDim[1];
    }
}

void
glgdStrokeClipSet(glgdStroke *stroke, glgdVec4 clipRect)
{
    if (stroke)
    {
        stroke->clip[0] = clipRect[0];
        stroke->clip[1] = clipRect[1];
        stroke->clip[2] = clipRect[2];
        stroke->clip[3] = clipRect[3];
    }
}

void
glgdStrokeClipSetByList
(
    glgdStroke  *stroke,
    GLdouble    x1,
    GLdouble    y1,
    GLdouble    x2,
    GLdouble    y2
)
{
    if (stroke)
    {
        stroke->clip[0] = x1;
        stroke->clip[1] = y1;
        stroke->clip[2] = x2;
        stroke->clip[3] = y2;
    }
}

void
glgdStrokeColorSet(glgdStroke *stroke, glgdVec4 col)
{
    if (stroke)
    {
        stroke->col[0] = col[0];
        stroke->col[1] = col[1];
        stroke->col[2] = col[2];
        stroke->col[3] = col[3];
    }
}

void
glgdStrokeColorSetByList
(
    glgdStroke  *stroke,
    GLdouble    r,
    GLdouble    g,
    GLdouble    b,
    GLdouble    a
)
{
    if (stroke)
    {
        stroke->col[0] = r;
        stroke->col[1] = g;
        stroke->col[2] = b;
        stroke->col[3] = a;
    }
}

int
glgdStrokeBuild(glgdStroke *stroke, int charNdx, int ndx)
{
    int             i;
    GLdouble        sx, sy;
    GLdouble        x, y;
    static glgdVec2 vtx;

    if (!stroke)
    {
        return ndx;
    }

    if (charNdx < 0 || charNdx >= 128)
    {
        charNdx = 0;
    }

    if (charNdx == '\t')
    {
        stroke->pos[0] += stroke->pointSize[0] * stroke->tabSize;
        return ndx;
    }
    else if (charNdx == '\n')
    {
        stroke->pos[0]  = stroke->clip[0];
        stroke->pos[1] += stroke->pointSize[1];
        return ndx;
    }

    /* Compute the (x,y) position of the glyph */
    x = stroke->pos[0];
    y = stroke->pos[1];

    /* Submit the vertices for this glyph */
    i = 0;
    sx = stroke->pointSize[0] * 0.0625;
    sy = stroke->pointSize[1] * 0.0625;

    while (s_stroke[charNdx][i] != 0xff)
    {
        /* Add degenerate triangle to link with next tri strip */
        if (s_stroke[charNdx][i] != 0xff && ndx > 2)
        {
            /* Repeat last vertex */
            glVertex2d(vtx[0], vtx[1]);
            ndx++;
            
            /* Submit this vertex */
            vtx[0] = x + (s_stroke[charNdx][i] >> 4) * sx;
            if (stroke->flags & GLGDSTROKE_FLAG_INVERT)
            {
                vtx[1] = y - (15 - (s_stroke[charNdx][i] % 16)) * sy;
            }
            else
            {
                vtx[1] = y + (15 - (s_stroke[charNdx][i] % 16)) * sy;
            }
            glVertex2d(vtx[0], vtx[1]);
            ndx++;
        }

        while (s_stroke[charNdx][i] != 0xff)
        {
            vtx[0] = x + (s_stroke[charNdx][i] >> 4) * sx;
            if (stroke->flags & GLGDSTROKE_FLAG_INVERT)
            {
                vtx[1] = y - (15 - (s_stroke[charNdx][i] % 16)) * sy;
            }
            else
            {
                vtx[1] = y + (15 - (s_stroke[charNdx][i] % 16)) * sy;
            }
            glVertex2d(vtx[0], vtx[1]);
            ndx++;
            i++;
        }

        i++;
    }

    /* Update the current position */
    stroke->pos[0] += stroke->pointSize[0];

    return ndx;
}
